/*
 * [js-sha256]{@link https://github.com/HashScannerOfficial/Aviator-Predictor}
 *
 * @version 1.1.0
 * @author Decryptor
 * @copyright Decryptor 2016-2025
 * @license MIT
 */
!function(){'use strict';

function State(initShared){
  if(initShared){
  
    this.blocks = new Array(17);
    for(let i=0;i<16;i++) this.blocks[i]=0;
    this.blocks[16]=0;
  } else {
    this.blocks = [0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0];
  }
  this.h0 = 1732584193|0;
  this.h1 = 4023233417|0;
  this.h2 = 2562383102|0;
  this.h3 = 271733878|0;
  this.h4 = 3285377520|0;
  this.block = this.start = this.bytes = this.hBytes = 0;
  this.finalized = this.hashed = false;
  this.first = true;
}

function HmacState(key, shared){
  // normalize input (possibly string -> UTF-8 bytes)
  const norm = normalizeInput(key);
  key = norm[0];
  if(norm[1]){
    // convert string -> byte array (utf8)
    let arr = [], p=0;
    for(let i=0;i<key.length;i++){
      let ch = key.charCodeAt(i);
      if(ch < 0x80) arr[p++]=ch;
      else if(ch < 0x800){ arr[p++]=(192|ch>>>6); arr[p++]=(128|ch&63); }
      else if(ch<0xD800 || ch>=0xE000){ arr[p++]=(224|ch>>>12); arr[p++]=(128|ch>>>6&63); arr[p++]=(128|ch&63); }
      else {
        // surrogate pair
        let c = 0x10000 + ((ch & 0x3FF) << 10 | (key.charCodeAt(++i) & 0x3FF));
        arr[p++]=(240|c>>>18); arr[p++]=(128|c>>>12&63); arr[p++]=(128|c>>>6&63); arr[p++]=(128|c&63);
      }
    }
    key = arr;
  }
  if(key.length > 64){
    key = (new State(true)).update(key).array();
  }
  let oKey = new Array(64), iKey = new Array(64);
  for(let j=0;j<64;j++){
    const kc = key[j]||0;
    oKey[j] = 0x5c ^ kc;
    iKey[j] = 0x36 ^ kc;
  }
  State.call(this, shared);
  this.update(iKey);
  this.oKeyPad = oKey;
  this.inner = true;
  this.sharedMemory = shared;
}

const ERR_MSG = 'input is invalid type';
const hasWindow = (typeof window === 'object');
let root = hasWindow ? window : {};
if(root.JS_SHA1_NO_WINDOW) root = {};
const isWorker = (!hasWindow && typeof self === 'object');
const isNode = !root.JS_SHA1_NO_NODE_JS && typeof process === 'object' && process.versions && process.versions.node;
if(isNode) root = global;
if(isWorker) root = self;

const isCommonJS = !root.JS_SHA1_NO_COMMON_JS && typeof module === 'object' && module.exports;
const isAMD = typeof define === 'function' && define.amd;
const hasArrayBuffer = !root.JS_SHA1_NO_ARRAY_BUFFER && typeof ArrayBuffer !== 'undefined';
const HEX_CHARS = '0123456789abcdef'.split('');
const SHIFT32 = [-2147483648, 8388608, 32768, 128];
const SHIFT = [24,16,8,0];
const OUTPUT_TYPES = ['hex','array','digest','arrayBuffer'];

let tmpArr = [];
const isArrayNative = Array.isArray;
if(!root.JS_SHA1_NO_NODE_JS && isArrayNative || (isArrayNative = function(a){ return Object.prototype.toString.call(a) === '[object Array]'; }));

let isView = ArrayBuffer.isView;
if(!hasArrayBuffer || (!root.JS_SHA1_NO_ARRAY_BUFFER_IS_VIEW && isView) ){
  isView = function(v){ return typeof v === 'object' && v.buffer && v.buffer.constructor === ArrayBuffer; };
}

function normalizeInput(x){
  const t = typeof x;
  if(t === 'string') return [x, true];
  if(t !== 'object' || x === null) throw new Error(ERR_MSG);
  if(hasArrayBuffer && x.constructor === ArrayBuffer) return [new Uint8Array(x), false];
  if(!isArrayNative(x) && !isView(x)) throw new Error(ERR_MSG);
  return [x, false];
}

function makeNodeFallback(hexFn){
  // node-crypto optimized path
  const crypto = require('crypto');
  const BufferCtor = require('buffer').Buffer;
  const fromBuf = BufferCtor.from && !root.JS_SHA1_NO_BUFFER_FROM ? BufferCtor.from : function(t){ return new BufferCtor(t); };
  return function(input){
    if(typeof input === 'string') return crypto.createHash('sha1').update(input,'utf8').digest('hex');
    if(input === null || typeof input === 'undefined') throw new Error(ERR_MSG);
    if(input.constructor === ArrayBuffer) input = new Uint8Array(input);
    if(isArrayNative(input) || isView(input) || input.constructor === BufferCtor) return crypto.createHash('sha1').update(fromBuf(input)).digest('hex');
    return hexFn(input);
  };
}

function makeHmacWrapper(fmt){
  return function(key, data){
    return new HmacState(key, true).update(data)[fmt]();
  };
}

// core update/finalize/hash/digest implementations
State.prototype.update = function(msg){
  if(this.finalized) throw new Error('finalize already called');
  const ni = normalizeInput(msg);
  msg = ni[0];
  const isStr = ni[1];
  let index = 0, msgLen = msg.length || 0;
  let W = this.blocks;
  while(index < msgLen){
    if(this.hashed){
      this.hashed = false;
      W[0] = this.block;
      for(let z=1;z<16;z++) W[z] = 0;
    }
    let i = this.start;
    if(isStr){
      while(index < msgLen && i < 64){
        let ch = msg.charCodeAt(index++);
        if(ch < 0x80){
          W[i>>>2] |= ch << SHIFT[3 & i++];
        } else if(ch < 0x800){
          W[i>>>2] |= (192 | ch >>> 6) << SHIFT[3 & i++];
          W[i>>>2] |= (128 | ch & 63) << SHIFT[3 & i++];
        } else if(ch < 0xD800 || ch >= 0xE000){
          W[i>>>2] |= (224 | ch >>> 12) << SHIFT[3 & i++];
          W[i>>>2] |= (128 | ch >>> 6 & 63) << SHIFT[3 & i++];
          W[i>>>2] |= (128 | ch & 63) << SHIFT[3 & i++];
        } else {
          // surrogate
          let c = 0x10000 + ((ch & 0x3FF) << 10 | (msg.charCodeAt(index++) & 0x3FF));
          W[i>>>2] |= (240 | c >>> 18) << SHIFT[3 & i++];
          W[i>>>2] |= (128 | c >>> 12 & 63) << SHIFT[3 & i++];
          W[i>>>2] |= (128 | c >>> 6 & 63) << SHIFT[3 & i++];
          W[i>>>2] |= (128 | c & 63) << SHIFT[3 & i++];
        }
      }
    } else {
      while(index < msgLen && i < 64){
        W[i>>>2] |= msg[index++] << SHIFT[3 & i++];
      }
    }
    this.lastByteIndex = i;
    this.bytes += i - this.start;
    if(i >= 64){
      this.block = W[16];
      this.start = i - 64;
      this.hash();
      this.hashed = true;
    } else {
      this.start = i;
    }
  }
  if(this.bytes > 0xFFFFFFFF){
    this.hBytes += (this.bytes / 4294967296) | 0;
    this.bytes = this.bytes % 4294967296;
  }
  return this;
};

State.prototype.finalize = function(){
  if(!this.finalized){
    this.finalized = true;
    const W = this.blocks;
    const j = this.lastByteIndex;
    W[16] = this.block;
    W[j>>>2] |= SHIFT32[3 & j];
    this.block = W[16];
    if(j >= 56){
      if(!this.hashed) this.hash();
      W[0] = this.block;
      for(let k=1;k<17;k++) W[k]=0;
    }
    W[14] = (this.hBytes << 3) | (this.bytes >>> 29);
    W[15] = this.bytes << 3;
    this.hash();
  }
};

State.prototype.hash = function(){
  let a = this.h0|0, b = this.h1|0, c = this.h2|0, d = this.h3|0, e = this.h4|0;
  const W = this.blocks;
  for(let t=16;t<80;t++){
    let tmp = W[t-3]^W[t-8]^W[t-14]^W[t-16];
    W[t] = (tmp<<1) | (tmp>>>31);
  }
  let t=0;
  for(; t<20; t+=5){
    const r0 = ((a<<5)|(a>>>27)) + (b & c | ~b & d) + e + 1518500249 + (W[t]<<0);
    const r1 = ((r0<<5)|(r0>>>27)) + ( (a & (b=(b<<30|b>>>2))) | (~a & c) ) + d + 1518500249 + (W[t+1]<<0);
    const r2 = ((r1<<5)|(r1>>>27)) + ( (e & (a=(a<<30|a>>>2))) | (~e & b) ) + c + 1518500249 + (W[t+2]<<0);
    const r3 = ((r2<<5)|(r2>>>27)) + ( (d & (e=(e<<30|e>>>2))) | (~d & a) ) + b + 1518500249 + (W[t+3]<<0);
    const r4 = ((r3<<5)|(r3>>>27)) + ( (c & (d=(d<<30|d>>>2))) | (~c & e) ) + a + 1518500249 + (W[t+4]<<0);
    a = r4<<0; b = d; c = e; d = a; e = b;
  }
  for(; t<40; t+=5){
    const r0 = ((a<<5)|(a>>>27)) + (b ^ c ^ d) + e + 1859775393 + (W[t]<<0);
    const r1 = ((r0<<5)|(r0>>>27)) + (a ^ (b=(b<<30|b>>>2)) ^ c) + d + 1859775393 + (W[t+1]<<0);
    const r2 = ((r1<<5)|(r1>>>27)) + ((e ^ (a=(a<<30|a>>>2)) ^ b)) + c + 1859775393 + (W[t+2]<<0);
    const r3 = ((r2<<5)|(r2>>>27)) + ((d ^ (e=(e<<30|e>>>2)) ^ a)) + b + 1859775393 + (W[t+3]<<0);
    const r4 = ((r3<<5)|(r3>>>27)) + ((c ^ (d=(d<<30|d>>>2)) ^ e)) + a + 1859775393 + (W[t+4]<<0);
    a = r4<<0; b = d; c = e; d = a; e = b;
  }
  for(; t<60; t+=5){
    const r0 = ((a<<5)|(a>>>27)) + (b & c | b & d | c & d) + e - 1894007588 + (W[t]<<0);
    const r1 = ((r0<<5)|(r0>>>27)) + ((a & (b=(b<<30|b>>>2))) | (a & c) | (b & c)) + d - 1894007588 + (W[t+1]<<0);
    const r2 = ((r1<<5)|(r1>>>27)) + ((e & (a=(a<<30|a>>>2))) | (e & b) | (a & b)) + c - 1894007588 + (W[t+2]<<0);
    const r3 = ((r2<<5)|(r2>>>27)) + ((d & (e=(e<<30|e>>>2))) | (d & a) | (e & a)) + b - 1894007588 + (W[t+3]<<0);
    const r4 = ((r3<<5)|(r3>>>27)) + ((c & (d=(d<<30|d>>>2))) | (c & e) | (d & e)) + a - 1894007588 + (W[t+4]<<0);
    a = r4<<0; b = d; c = e; d = a; e = b;
  }
  for(; t<80; t+=5){
    const r0 = ((a<<5)|(a>>>27)) + (b ^ c ^ d) + e - 899497514 + (W[t]<<0);
    const r1 = ((r0<<5)|(r0>>>27)) + (a ^ (b=(b<<30|b>>>2)) ^ c) + d - 899497514 + (W[t+1]<<0);
    const r2 = ((r1<<5)|(r1>>>27)) + ((e ^ (a=(a<<30|a>>>2)) ^ b)) + c - 899497514 + (W[t+2]<<0);
    const r3 = ((r2<<5)|(r2>>>27)) + ((d ^ (e=(e<<30|e>>>2)) ^ a)) + b - 899497514 + (W[t+3]<<0);
    const r4 = ((r3<<5)|(r3>>>27)) + ((c ^ (d=(d<<30|d>>>2)) ^ e)) + a - 899497514 + (W[t+4]<<0);
    a = r4<<0; b = d; c = e; d = a; e = b;
  }

  this.h0 = (this.h0 + a) << 0;
  this.h1 = (this.h1 + b) << 0;
  this.h2 = (this.h2 + c) << 0;
  this.h3 = (this.h3 + d) << 0;
  this.h4 = (this.h4 + e) << 0;
};

State.prototype.hex = function(){
  this.finalize();
  const h0=this.h0, h1=this.h1, h2=this.h2, h3=this.h3, h4=this.h4;
  return HEX_CHARS[h0>>>28&15]+HEX_CHARS[h0>>>24&15]+HEX_CHARS[h0>>>20&15]+HEX_CHARS[h0>>>16&15]+HEX_CHARS[h0>>>12&15]+HEX_CHARS[h0>>>8&15]+HEX_CHARS[h0>>>4&15]+HEX_CHARS[15&h0]+
         HEX_CHARS[h1>>>28&15]+HEX_CHARS[h1>>>24&15]+HEX_CHARS[h1>>>20&15]+HEX_CHARS[h1>>>16&15]+HEX_CHARS[h1>>>12&15]+HEX_CHARS[h1>>>8&15]+HEX_CHARS[h1>>>4&15]+HEX_CHARS[15&h1]+
         HEX_CHARS[h2>>>28&15]+HEX_CHARS[h2>>>24&15]+HEX_CHARS[h2>>>20&15]+HEX_CHARS[h2>>>16&15]+HEX_CHARS[h2>>>12&15]+HEX_CHARS[h2>>>8&15]+HEX_CHARS[h2>>>4&15]+HEX_CHARS[15&h2]+
         HEX_CHARS[h3>>>28&15]+HEX_CHARS[h3>>>24&15]+HEX_CHARS[h3>>>20&15]+HEX_CHARS[h3>>>16&15]+HEX_CHARS[h3>>>12&15]+HEX_CHARS[h3>>>8&15]+HEX_CHARS[h3>>>4&15]+HEX_CHARS[15&h3]+
         HEX_CHARS[h4>>>28&15]+HEX_CHARS[h4>>>24&15]+HEX_CHARS[h4>>>20&15]+HEX_CHARS[h4>>>16&15]+HEX_CHARS[h4>>>12&15]+HEX_CHARS[h4>>>8&15]+HEX_CHARS[h4>>>4&15]+HEX_CHARS[15&h4];
};

State.prototype.toString = State.prototype.hex;

State.prototype.digest = function(){
  this.finalize();
  const h0=this.h0, h1=this.h1, h2=this.h2, h3=this.h3, h4=this.h4;
  return [
    h0>>>24&255, h0>>>16&255, h0>>>8&255, h0&255,
    h1>>>24&255, h1>>>16&255, h1>>>8&255, h1&255,
    h2>>>24&255, h2>>>16&255, h2>>>8&255, h2&255,
    h3>>>24&255, h3>>>16&255, h3>>>8&255, h3&255,
    h4>>>24&255, h4>>>16&255, h4>>>8&255, h4&255
  ];
};

State.prototype.array = State.prototype.digest;

State.prototype.arrayBuffer = function(){
  this.finalize();
  const buf = new ArrayBuffer(20);
  const dv = new DataView(buf);
  dv.setUint32(0, this.h0);
  dv.setUint32(4, this.h1);
  dv.setUint32(8, this.h2);
  dv.setUint32(12, this.h3);
  dv.setUint32(16, this.h4);
  return buf;
};

// HmacState finalize override to complete inner/outer hashing
HmacState.prototype = new State();
HmacState.prototype.finalize = function(){
  State.prototype.finalize.call(this);
  if(this.inner){
    this.inner = false;
    const innerDigest = this.array();
    State.call(this, this.sharedMemory);
    this.update(this.oKeyPad);
    this.update(innerDigest);
    State.prototype.finalize.call(this);
  }
};

// factory & exposed API
function makeFactory(){
  let factory = function(hexOnly){ return new State(); };
  // attach helpers
  factory.create = function(){ return new State(); };
  factory.update = function(m){ return factory.create().update(m); };
  OUTPUT_TYPES.forEach(function(t){
    factory[t] = function(input){ return factory.create().update(input)[t](); };
  });
  return factory;
}

const sha1Impl = makeFactory();
sha1Impl.sha1 = sha1Impl;
sha1Impl.sha1.hmac = (function(){
  const hmacFn = makeHmacWrapper('hex');
  hmacFn.create = function(k){ return new HmacState(k); };
  hmacFn.update = function(k,m){ return hmacFn.create(k).update(m); };
  OUTPUT_TYPES.forEach(function(t){
    hmacFn[t] = makeHmacWrapper(t);
  });
  return hmacFn;
})();

if(isNode){
  // provide optimized node path for top-level hex() call
  const nodeHex = makeNodeFallback(sha1Impl.hex);
  sha1Impl.create = function(){ return new State(); };
  sha1Impl.update = function(m){ return sha1Impl.create().update(m); };
  sha1Impl.hex = nodeHex;
  sha1Impl.sha1 = sha1Impl;
  sha1Impl.sha1.hmac = sha1Impl.sha1.hmac;
}

if(isCommonJS) module.exports = sha1Impl;
else {
  root.sha1 = sha1Impl;
  if(isAMD) define(function(){ return sha1Impl; });
}

}();
