#!/bin/bash
# unlocker.command - auto unlocker for .app bundles in the same folder
# Double-click this file in Finder (or drag an .app onto it) to remove quarantine and open the app.

# Get the directory where this script lives (works when double-clicked)
DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"

# If user passed one or more args (drag-drop), use those instead of auto-search
if [ $# -gt 0 ]; then
  targets=()
  for a in "$@"; do
    targets+=("$a")
  done
else
  # Find .app bundles in the same folder (only top-level)
  targets=()
  while IFS= read -r -d $'\0' f; do
    targets+=("$f")
  done < <(find "$DIR" -maxdepth 1 -type d -name "*.app" -print0)
fi

# Helper to show macOS dialog
show_dialog() {
  /usr/bin/osascript -e "display dialog $1 buttons {\"OK\"} with title \"Unlocker\""
}

# No apps found
if [ ${#targets[@]} -eq 0 ]; then
  /usr/bin/osascript -e 'tell app "Finder" to display dialog "No .app found in the same folder. Please place your .app (e.g. HashScanner.app) in the folder with this Unlocker and double-click again, or drag the .app onto this Unlocker." buttons {"OK"} with title "Unlocker"'
  exit 1
fi

# If exactly one target, use it automatically
if [ ${#targets[@]} -eq 1 ]; then
  chosen="${targets[0]}"
else
  # Multiple: ask user to choose via AppleScript list
  # Build a quoted list of basenames for Applescript
  applescript_list=""
  for f in "${targets[@]}"; do
    name="$(basename "$f")"
    # escape backslashes and double quotes
    esc_name="${name//\\/\\\\}"
    esc_name="${esc_name//\"/\\\"}"
    if [ -z "$applescript_list" ]; then
      applescript_list="\"$esc_name\""
    else
      applescript_list="$applescript_list, \"$esc_name\""
    fi
  done

  chosen_name=$(/usr/bin/osascript <<AS
set theList to { $applescript_list }
set chosen to choose from list theList with prompt "Choose the app to unlock and open:" default items {item 1 of theList}
if chosen is false then
  return "CANCEL"
else
  return item 1 of chosen
end if
AS
)

  if [ "$chosen_name" = "CANCEL" ]; then
    /usr/bin/osascript -e 'display dialog "No app selected. Exiting." buttons {"OK"} with title "Unlocker"'
    exit 0
  fi

  # find the full path that matches the chosen base name
  chosen=""
  for f in "${targets[@]}"; do
    if [ "$(basename "$f")" = "$chosen_name" ]; then
      chosen="$f"
      break
    fi
  done

  if [ -z "$chosen" ]; then
    /usr/bin/osascript -e 'display dialog "Selected app not found. Exiting." buttons {"OK"} with title "Unlocker"'
    exit 1
  fi
fi

# Final safety check: chosen is a directory and ends with .app
if [ ! -d "$chosen" ]; then
  /usr/bin/osascript -e 'display dialog "The selected item is not an application. Exiting." buttons {"OK"} with title "Unlocker"'
  exit 1
fi

# Run xattr -cr and open. Both commands are absolute paths to avoid PATH issues.
/usr/bin/printf "Removing quarantine from: %s\n" "$chosen"
/usr/bin/xattr -cr "$chosen" 2>/dev/null
ret=$?
if [ $ret -ne 0 ]; then
  /usr/bin/osascript -e 'display dialog "Failed to remove quarantine (permission or other error). You can try running this script from Terminal for more details." buttons {"OK"} with title "Unlocker"'
  # still attempt to open so user can try
fi

/usr/bin/printf "Opening: %s\n" "$chosen"
/usr/bin/open "$chosen"

# Let user know we're done
/usr/bin/osascript -e 'display dialog "Done — the app was unlocked (if possible) and opened." buttons {"OK"} with title "Unlocker"'
exit 0
